//
//  OpenGLClasses.m
//  OpenGLESwoIB
//
//  Created by 渡部 心 on 2013/04/18.
//  Copyright (c) 2013年 渡部 心. All rights reserved.
//

#import "OpenGLClasses.h"
#import <OpenGLES/ES2/glext.h>

void glcIdentityMatrix(float* matrix) {
    matrix[ 0] = matrix[ 5] = matrix[10] = matrix[15] = 1.0f;
    matrix[ 1] = matrix[ 2] = matrix[ 3] = matrix[ 4] =
    matrix[ 6] = matrix[ 7] = matrix[ 8] = matrix[ 9] =
    matrix[11] = matrix[12] = matrix[13] = matrix[14] = 0.0f;
}

void glcOrthogonalMatrix(float left, float right, float bottom, float top, float near, float far, float* matrix) {
    float dx = right - left;
    float dy = top - bottom;
    float dz = far - near;
    
    matrix[ 0] = 2.0f / dx;
    matrix[ 5] = 2.0f / dy;
    matrix[10] = -2.0f / dz;
    matrix[12] = -(left + right) / dx;
    matrix[13] = -(bottom + top) / dy;
    matrix[14] = -(near + far) / dz;
    matrix[15] = 1.0f;
    matrix[ 1] = matrix[ 2] = matrix[ 3] = matrix[ 4] =
    matrix[ 6] = matrix[ 7] = matrix[ 8] = matrix[ 9] =
    matrix[11] = 0.0f;
}

void glcPerspectiveMatrix(float left, float right, float bottom, float top, float near, float far, float* matrix) {
    float dx = right - left;
    float dy = top - bottom;
    float dz = far - near;
    
    matrix[ 0] = 2.0f * near / dx;
    matrix[ 5] = 2.0f * near / dy;
    matrix[ 8] = (left + right) / dx;
    matrix[ 9] = (bottom + top) / dy;
    matrix[10] = -(near + far) / dz;
    matrix[11] = -1.0f;
    matrix[14] = -2.0f * far * near / dz;
    matrix[ 1] = matrix[ 2] = matrix[ 3] = matrix[ 4] = matrix[ 6] =
    matrix[ 7] = matrix[12] = matrix[13] = matrix[15] = 0.0f;
}

void glcLookAt(float ex, float ey, float ez, float tx, float ty, float tz, float ux, float uy, float uz, float* matrix) {
    tx = ex - tx;
    ty = ey - ty;
    tz = ez - tz;
    float l = sqrtf(tx * tx + ty * ty + tz * tz);
    matrix[ 2] = tx / l;
    matrix[ 6] = ty / l;
    matrix[10] = tz / l;
    
    tx = uy * matrix[10] - uz * matrix[ 6];
    ty = uz * matrix[ 2] - ux * matrix[10];
    tz = ux * matrix[ 6] - uy * matrix[ 2];
    
    l = sqrtf(tx * tx + ty * ty + tz * tz);
    matrix[ 0] = tx / l;
    matrix[ 4] = ty / l;
    matrix[ 8] = tz / l;
    
    matrix[ 1] = matrix[ 6] * matrix[ 8] - matrix[10] * matrix[ 4];
    matrix[ 5] = matrix[10] * matrix[ 0] - matrix[ 2] * matrix[ 8];
    matrix[ 9] = matrix[ 2] * matrix[ 4] - matrix[ 6] * matrix[ 0];
    
    matrix[12] = -(ex * matrix[ 0] + ey * matrix[ 4] + ez * matrix[ 8]);
    matrix[13] = -(ex * matrix[ 1] + ey * matrix[ 5] + ez * matrix[ 9]);
    matrix[14] = -(ex * matrix[ 2] + ey * matrix[ 6] + ez * matrix[10]);
    
    matrix[ 3] = matrix[ 7] = matrix[11] = 0.0f;
    matrix[15] = 1.0f;
}

void glcScaleMatrix(float sx, float sy, float sz, float* matrix) {
    matrix[ 0] = sx;
    matrix[ 5] = sy;
    matrix[10] = sz;
    matrix[15] = 1.0f;
    matrix[ 1] = matrix[ 2] = matrix[ 3] = matrix[ 4] =
    matrix[ 6] = matrix[ 7] = matrix[ 8] = matrix[ 9] =
    matrix[11] = matrix[12] = matrix[13] = matrix[14] = 0.0f;
}

void glcRotateMatrix(float angle, float rx, float ry, float rz, float* matrix) {
    float length = sqrtf(rx * rx + ry * ry + rz * rz);
    rx /= length;
    ry /= length;
    rz /= length;
    float c = cosf(angle);
    float s = sinf(angle);
    float oneMinusC = 1 - c;
    
    matrix[ 0] = rx * rx * oneMinusC + c;
    matrix[ 1] = rx * ry * oneMinusC + rz * s;
    matrix[ 2] = rz * rx * oneMinusC - ry * s;
    matrix[ 4] = rx * ry * oneMinusC - rz * s;
    matrix[ 5] = ry * ry * oneMinusC + c;
    matrix[ 6] = ry * rz * oneMinusC + rx * s;
    matrix[ 8] = rz * rx * oneMinusC + ry * s;
    matrix[ 9] = ry * rz * oneMinusC - rx * s;
    matrix[10] = rz * rz * oneMinusC + c;
    
    matrix[ 3] = matrix[ 7] = matrix[11] =
    matrix[12] = matrix[13] = matrix[14] = 0.0f;
    matrix[15] = 1.0f;
}

void glcTranslateMatrix(float tx, float ty, float tz, float* matrix) {
    matrix[ 0] = matrix[ 5] = matrix[10] = matrix[15] = 1.0f;
    matrix[ 1] = matrix[ 2] = matrix[ 3] = matrix[ 4] =
    matrix[ 6] = matrix[ 7] = matrix[ 8] = matrix[ 9] =
    matrix[11] = 0.0f;
    matrix[12] = tx;
    matrix[13] = ty;
    matrix[14] = tz;
}

void glcCopyMatrix(const float* m0, float* matrix) {
    for (int i = 0; i < 16; i++) {
        matrix[i] = m0[i];
    }
}

void glcTransposeMatrix(const float* m0, float* matrix) {
    matrix[ 0] = m0[ 0];//  0  4  8 12
    matrix[ 5] = m0[ 5];//  1  5  9 13
    matrix[10] = m0[10];//  2  6 10 14
    matrix[15] = m0[15];//  3  7 11 15
    
    matrix[ 1] = m0[ 4];
    matrix[ 4] = m0[ 1];
    matrix[ 2] = m0[ 8];
    matrix[ 8] = m0[ 2];
    matrix[ 3] = m0[12];
    matrix[12] = m0[ 3];
    
    matrix[ 6] = m0[ 9];
    matrix[ 9] = m0[ 6];
    matrix[ 7] = m0[13];
    matrix[13] = m0[ 7];
    
    matrix[11] = m0[14];
    matrix[14] = m0[11];
}

void glcTransposeMatrixSelf(float* matrix) {
    float temp[16];
    for (int i = 0; i < 16; i++) {
        temp[i] = matrix[i];
    }
    
    matrix[ 0] = temp[ 0];//  0  4  8 12
    matrix[ 5] = temp[ 5];//  1  5  9 13
    matrix[10] = temp[10];//  2  6 10 14
    matrix[15] = temp[15];//  3  7 11 15
    
    matrix[ 1] = temp[ 4];
    matrix[ 4] = temp[ 1];
    matrix[ 2] = temp[ 8];
    matrix[ 8] = temp[ 2];
    matrix[ 3] = temp[12];
    matrix[12] = temp[ 3];
    
    matrix[ 6] = temp[ 9];
    matrix[ 9] = temp[ 6];
    matrix[ 7] = temp[13];
    matrix[13] = temp[ 7];
    
    matrix[11] = temp[14];
    matrix[14] = temp[11];
}

void glcMultiplyMatrix(const float* m0, const float* m1, float* matrix) {
    for (int i = 0; i < 16; i++) {
        int j = i & 3;
        int k = i & ~3;
        matrix[i] = m0[ 0 + j] * m1[k + 0] + 
        m0[ 4 + j] * m1[k + 1] + 
        m0[ 8 + j] * m1[k + 2] + 
        m0[12 + j] * m1[k + 3];
    }
}

void glcPrintMatrix(const float* matrix) {
    for (int i = 0; i < 4; i++) {
        printf("%6.2g, %6.2g, %6.2g, %6.2g\n", matrix[0 + i], matrix[4 + i], matrix[8 + i], matrix[12 + i]);
    }
    printf("\n");
}

void glcPrintError(void) {
    GLenum error = glGetError();
    switch(error) {
        case GL_NO_ERROR:
            printf("OpenGL Error : No error\n");
            break;
        case GL_INVALID_ENUM:
            printf("OpenGL Error : Invalid enum\n");
            break;
        case GL_INVALID_VALUE:
            printf("OpenGL Error : Invalid value\n");
            break;
        case GL_INVALID_OPERATION:
            printf("OpenGL Error : Invalid operation\n");
            break;
        case GL_INVALID_FRAMEBUFFER_OPERATION:
            printf("OpenGL Error : Invalid framebuffer operation\n");
            break;
        case GL_OUT_OF_MEMORY:
            printf("OpenGL Error : Out of memory\n");
            break;
        default:
            printf("OpenGL Error : Unknown Error\n");
            break;
    }
}

//------------------------------------------------------------------------------------------------

const int maxPushDepth = 10;

@implementation MatrixStack

- (void)setCurrent:(GLfloat *)current {
    for (int i = 0; i < 16; ++i) {
        _current[i] = current[i];
    }
}

- (GLfloat*)current {
    return _current;
}

- (id)init {
    self = [super init];
	if (self) {
        _depth = maxPushDepth;
        _memoryPool = (GLfloat*)malloc(_depth * 16 * sizeof(GLfloat));
        _current = _memoryPool;
        glcIdentityMatrix(_current);
        _counter = 0;
	}
	
	return self;
}

- (id)initWithDepth:(unsigned int)depth {
    self = [super init];
	if (self) {
        _depth = depth;
        _memoryPool = (GLfloat*)malloc(_depth * 16 * sizeof(GLfloat));
        _current = _memoryPool;
        glcIdentityMatrix(_current);
        _counter = 0;
	}
	
	return self;
}

- (void)push {
    if (_counter < _depth - 1) {
        memcpy(_current + 16, _current, sizeof(GLfloat) * 16);
        _current += 16;
        ++_counter;
    }
}

- (void)pop {
    if (_counter > 0) {
        _current -= 16;
        --_counter;
    }
    else {
        glcIdentityMatrix(_current);
    }
}

- (void)resetLevel {
    if (_counter > 0) {
        memcpy(_current, _current - 16, sizeof(GLfloat) * 16);   
    }
    else {
        glcIdentityMatrix(_current);
    }
}

- (void)preMult:(GLfloat *)matrix {
    float temp[16];
    memcpy(temp, _current, sizeof(GLfloat) * 16);
    glcMultiplyMatrix(matrix, temp, _current);
}

- (void)postMult:(GLfloat *)matrix {
    float temp[16];
    memcpy(temp, _current, sizeof(GLfloat) * 16);
    glcMultiplyMatrix(temp, matrix, _current);
}

- (void)dealloc {
    free(_memoryPool);
}

@end

//------------------------------------------------------------------------------------------------

@interface Shader (PrivateMethods)

- (BOOL)CreateShader;

@end

@implementation Shader

- (id)init {
    self = [super init];
	if (self) {
        _programID = 0;
        _VSID = 0;
        _FSID = 0;
        _attributes = [[NSMutableDictionary alloc] initWithCapacity:5];
        _uniforms = [[NSMutableDictionary alloc] initWithCapacity:5];
	}
	
	return self;
}

- (id)initWithShaderPath_VS:(NSString*)vspath FS:(NSString*)fspath {
    self = [self init];
	if (self) {
        _VSPath = [vspath copy];
        _FSPath = [fspath copy];
        if ([_VSPath isEqualToString:@""] == NO) {
            _VSSource = [[NSString alloc] initWithContentsOfFile:[[NSBundle mainBundle] pathForAuxiliaryExecutable:_VSPath] 
                                                         encoding:NSUTF8StringEncoding error:nil];
        }
        if ([_FSPath isEqualToString:@""] == NO) {
            _FSSource = [[NSString alloc] initWithContentsOfFile:[[NSBundle mainBundle] pathForAuxiliaryExecutable:_FSPath] 
                                                         encoding:NSUTF8StringEncoding error:nil];
        }
        _compiled = [self CreateShader];
    }
    
	return self;
}

- (void)dealloc {
    if (_VSID) {
        glDeleteShader(_VSID);
    }
    if (_FSID) {
        glDeleteShader(_FSID);
    }
    if (_programID) {
        glDeleteProgram(_programID);
    }
}

- (BOOL)CreateShader {
    int status_code;
    GLint loglength;
    
    _VSID = glCreateShader(GL_VERTEX_SHADER);
    
    const GLchar* vssource = (GLchar*)[_VSSource UTF8String];
    
    glShaderSource(_VSID, 1, &vssource, NULL);
    glCompileShader(_VSID);
    glGetShaderiv(_VSID, GL_INFO_LOG_LENGTH, &loglength);
    if (loglength > 0) {
        GLchar* log = (GLchar*)malloc(loglength * sizeof(GLchar));
        glGetShaderInfoLog(_VSID, loglength, &loglength, log);
        printf("Vertex Shader Compile Error\n%s\n", log);
        free(log);
    }
    glGetShaderiv(_VSID, GL_COMPILE_STATUS, &status_code);
    if (status_code == 0)
    {
        glDeleteShader(_VSID);
        return FALSE;
    }
    
    _FSID = glCreateShader(GL_FRAGMENT_SHADER);
    
    const GLchar* fssource = (GLchar*)[_FSSource UTF8String];
    
    glShaderSource(_FSID, 1, &fssource, NULL);
    glCompileShader(_FSID);
    glGetShaderiv(_FSID, GL_INFO_LOG_LENGTH, &loglength);
    if (loglength > 0) {
        GLchar* log = (GLchar*)malloc(loglength * sizeof(GLchar));
        glGetShaderInfoLog(_FSID, loglength, &loglength, log);
        printf("Fragment Shader Compile Error\n%s\n", log);
        free(log);
    }
    glGetShaderiv(_FSID, GL_COMPILE_STATUS, &status_code);
    if (status_code == 0)
    {
        glDeleteShader(_VSID);
        glDeleteShader(_FSID);
        return FALSE;
    }
    
    _programID = glCreateProgram();
    glAttachShader(_programID, _FSID);
    glAttachShader(_programID, _VSID);
    
    glLinkProgram(_programID);
    glGetProgramiv(_programID, GL_INFO_LOG_LENGTH, &loglength);
    if (loglength > 0) {
        GLchar* log = (GLchar*)malloc(loglength * sizeof(GLchar));
        glGetProgramInfoLog(_programID, loglength, &loglength, log);
        printf("%s\n", log);
        free(log);
    }
    glGetProgramiv(_programID, GL_LINK_STATUS, &status_code);
    if (status_code == 0) {
        glDeleteShader(_VSID);
        glDeleteShader(_FSID);
        glDeleteProgram(_programID);
        return FALSE;
    }
    
    glUseProgram(_programID);
    
    [self validate];
    
    return TRUE;
}

- (BOOL)compile {
    if (_VSID) {
        glDeleteShader(_VSID);
    }
    if (_FSID) {
        glDeleteShader(_FSID);
    }
    if (_programID) {
        glDeleteProgram(_programID);
    }
    return _compiled = [self CreateShader];
}

- (void)validate {
    int status_code;
    int loglength;
    glValidateProgram(_programID);
    glGetProgramiv(_programID, GL_INFO_LOG_LENGTH, &loglength);
    if (loglength > 0) {
        GLchar* log = (GLchar*)malloc(loglength * sizeof(GLchar));
        glGetProgramInfoLog(_programID, loglength, &loglength, log);
        printf("%s", log);
        free(log);
    }
    glGetProgramiv(_programID, GL_VALIDATE_STATUS, &status_code);
    if (status_code == 0) {
        printf("status_code : GL_FALSE\n");
    }
}

- (void)deleteShader {
    glDeleteShader(_VSID);
    glDeleteShader(_FSID);
    glDeleteProgram(_programID);
    _VSID = _FSID = _programID = 0;
}

- (void)useProgram {
    glUseProgram(_programID);
}

- (void)validateAttribute:(NSString*)attribute {
    if (_compiled) {
        GLuint location = glGetAttribLocation(_programID, attribute.UTF8String);
        if (location != -1) {
            [_attributes setObject:[NSNumber numberWithUnsignedInt:location] forKey:attribute];
        }
    }
}

- (void)validateUniform:(NSString*)uniform {
    if (_compiled) {
        GLuint location = glGetUniformLocation(_programID, uniform.UTF8String);
        if (location != -1) {
            [_uniforms setObject:[NSNumber numberWithUnsignedInt:location] forKey:uniform];
        }
    }
}

- (GLuint)attribute:(NSString*)attribute {
    return [(NSNumber*)[_attributes objectForKey:attribute] unsignedIntValue];
}

- (GLuint)uniform:(NSString*)uniform {
    return [(NSNumber*)[_uniforms objectForKey:uniform] unsignedIntValue];
}

- (void)enableVertexAttribArray:(NSString *)attribute {
    glEnableVertexAttribArray([(NSNumber*)[_attributes objectForKey:attribute] unsignedIntValue]);
}

- (void)vertexAttribPointer:(NSString *)attribute numComps:(GLint)nc type:(GLenum)t 
                 normalized:(GLboolean)norm stride:(GLsizei)s pointer:(GLvoid *)ptr {
    glVertexAttribPointer([(NSNumber*)[_attributes objectForKey:attribute] unsignedIntValue], nc, t, norm, s, ptr);
}

@end

//------------------------------------------------------------------------------------------------

@implementation BufferObject

- (id)initWithData:(void*)data size:(int)size length:(int)length Target:(BufferTarget)target Usage:(BufferUsageHint)usage {
    self = [super init];
	if (self) {
        _ID = 0;
        _target = target;
        _usage = usage;
        _unitSize = size;
        _length = length;
        
        glGenBuffers(1, &_ID);
        glBindBuffer(_target, _ID);
        glBufferData(_target, size * length, data, _usage);
        
        int BufferSize;
        glGetBufferParameteriv(_target, GL_BUFFER_SIZE, &BufferSize);
        glBindBuffer(_target, 0);
        if (length * size != BufferSize) {
            printf("Can't create VBO.\n");
            self = nil;
        }
	}
	
	return self;
}

- (void)bindBuffer {
    glBindBuffer(_target, _ID);
}

- (void)unbindBuffer {
    glBindBuffer(_target, 0);
}

- (void)deleteBuffer {
    glDeleteBuffers(1, &_ID);
    _ID = 0;
}

- (void)dealloc {
    glDeleteBuffers(1, &_ID);
}

- (void)bufferSubData:(void*)data offset:(long)offset size:(long)size {
    glBindBuffer(_target, _ID);
    glBufferSubData(_target, offset, size, data);
}

@end

//------------------------------------------------------------------------------------------------

@implementation VAOSet

- (id)init {
    self = [super init];
    if (self) {
        glGenVertexArraysOES(1, &_ID);
    }
    
    return self;
}

- (void)bindVertexArray {
    glBindVertexArrayOES(_ID);
}

- (void)deleteVertexArray {
    glDeleteVertexArraysOES(1, &_ID);
    _ID = 0;
}

- (void)dealloc {
    glDeleteVertexArraysOES(1, &_ID);
}

@end

//------------------------------------------------------------------------------------------------

@implementation FBO

- (id)initWithSizeW:(int)w H:(int)h colorRB:(BOOL)colorRB depthRB:(BOOL)depthRB
colorInternalFormat:(SizedInternalFormat)colorInternalFormat
depthInternalFormat:(SizedInternalFormat)depthInternalFormat {
    self = [super init];
    if (self) {
        glGenFramebuffers(1, &_FBOID);
        glBindFramebuffer(GL_FRAMEBUFFER, _FBOID);
        _colorIsRB = colorRB;
        _depthIsRB = depthRB;
        
        if (colorRB) {
            glGenRenderbuffers(1, &_colorBufferID);
            glBindRenderbuffer(GL_RENDERBUFFER, _colorBufferID);
            glRenderbufferStorage(GL_RENDERBUFFER, colorInternalFormat, w, h);
            glFramebufferRenderbuffer(GL_FRAMEBUFFER, Attachment_Color0, GL_RENDERBUFFER, _colorBufferID);
        }
        else {
            glGenTextures(1, &_colorBufferID);
            glBindTexture(GL_TEXTURE_2D, _colorBufferID);
            glTexImage2D(GL_TEXTURE_2D, 0, colorInternalFormat, w, h, 0, PixelFormat_RGBA, PixelType_Byte, NULL);
            glFramebufferTexture2D(GL_FRAMEBUFFER, Attachment_Color0, GL_TEXTURE_2D, _colorBufferID, 0);
        }
        
        if (depthRB) {
            glGenRenderbuffers(1, &_depthBufferID);
            glBindRenderbuffer(GL_RENDERBUFFER, _depthBufferID);
            glRenderbufferStorage(GL_RENDERBUFFER, depthInternalFormat, w, h);
            glFramebufferRenderbuffer(GL_FRAMEBUFFER, Attachment_Depth, GL_RENDERBUFFER, _depthBufferID);
        }
        else {
            glGenTextures(1, &_depthBufferID);
            glBindTexture(GL_TEXTURE_2D, _depthBufferID);
            glTexImage2D(GL_TEXTURE_2D, 0, depthInternalFormat, w, h, 0, GL_RGBA, GL_TEXTURE_2D, NULL);
            glFramebufferTexture2D(GL_FRAMEBUFFER, Attachment_Depth, GL_TEXTURE_2D, _depthBufferID, 0);
        }
        
        [self checkFrameBufferStatus];
        glBindFramebuffer(GL_FRAMEBUFFER, 0);
    }
    
    return self;
}

- (void)dealloc {
    if (_colorIsRB) {
        glDeleteRenderbuffers(1, &_colorBufferID);
    }
    else {
        glDeleteTextures(1, &_colorBufferID);
    }
    
    if (_depthIsRB) {
        glDeleteRenderbuffers(1, &_depthBufferID);
    }
    else {
        glDeleteTextures(1, &_depthBufferID);
    }
    
    glDeleteFramebuffers(1, &_FBOID);
}

- (BOOL)checkFrameBufferStatus {
    int currentFB;
    glGetIntegerv(GL_FRAMEBUFFER_BINDING, &currentFB);
    glBindFramebuffer(GL_FRAMEBUFFER, _FBOID);
    GLenum status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    switch(status)
    {
        case GL_FRAMEBUFFER_COMPLETE:
            printf("Framebuffer complete.\n");
            return true;
            
        case GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT:
            printf("[ERROR] Framebuffer incomplete: Attachment is NOT complete.\n");
            return false;
            
        case GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT:
            printf("[ERROR] Framebuffer incomplete: No image is attached to FBO.\n");
            return false;
            
        case GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS:
            printf("[ERROR] Framebuffer incomplete: Not all attached images have the same width and height.\n");
            return false;
            
        case GL_FRAMEBUFFER_UNSUPPORTED:
            printf("[ERROR] Unsupported by FBO implementation.\n");
            return false;
            
        default:
            printf("[ERROR] Unknown error.\n");
            return false;
    }
    glBindFramebuffer(GL_FRAMEBUFFER, currentFB);
}

@end

//------------------------------------------------------------------------------------------------
